//	TorusGamesKeyboardView.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesKeyboardView.h"
#import "GeometryGamesUtilities-Common.h"
#import "GeometryGamesUtilities-Mac-iOS.h"


#define	KOREAN_SHIFT_CHARACTER	u'⬆' // formerly u'⇧'
#define KOREAN_ENTER_CHARACTER	u'⏎' // formerly u'↵'
#define HANZI_REQUEST_KEY		0xFFFFFFFB
#define JAMO_KEY				0xFFFFFFFC
#define KOREAN_SHIFT_KEY		0xFFFFFFFD
#define KOREAN_ENTER_KEY		0xFFFFFFFE
#define GENERIC_KEY				0xFFFFFFFF

//	Re-package a 2-letter language code as an unsigned integer
//	with the code's letters in its two lower-order bytes.
#define LANGUAGE_CODE(a,b)	((uint_fast16_t)(((a) << 8) | (b)))


typedef struct
{
	unsigned int	itsNumRows,
					*itsRowLengths;
	double			itsFullWidth,	//	measured in key-widths (ignoring key insets)
					*itsRowStarts;	//	measured in key-widths (ignoring key insets)
	unichar			**itsRows;
} KeyboardLayout;


static KeyboardLayout
	gKeyboardDE =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'z', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'y', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardEL =
	{
		3,
		(unsigned int [3]) {8, 9, 7},
		9.0,
		(double [3]) {1.0, 0.0, 1.0},
		(unichar *[3])
		{
			(unichar [ 8]) {u'ε', u'ρ', u'τ', u'υ', u'θ', u'ι', u'ο', u'π'},
			(unichar [ 9]) {u'α', u'σ', u'δ', u'φ', u'γ', u'η', u'ξ', u'κ', u'λ'},
			(unichar [ 7]) {u'ζ', u'χ', u'ψ', u'ω', u'β', u'ν', u'μ'}
		}
	},
	gKeyboardEN =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardES =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardFI =
	{
		3,
		(unsigned int [3]) {10, 11, 7},
		11.5,
		(double [3]) {0.0, 0.5, 1.0},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [11]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l', u'ö', u'ä'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardFR =
	{
		3,
		(unsigned int [3]) {10, 10, 6},
		10.0,
		(double [3]) {0.0, 0.0, 2.0},
		(unichar *[3])
		{
			(unichar [10]) {u'a', u'z', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [10]) {u'q', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l', u'm'},
			(unichar [ 6]) {u'w', u'x', u'c', u'v', u'b', u'n'}
		}
	},
	gKeyboardIT =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardJA =
	{
		3,
		(unsigned int [3]) {5, 8, 8},
		10.0,
		(double [3]) {0.5, 1.0, 1.0},
		(unichar *[3])
		{
			(unichar [ 5]) {u'あ', u'い', u'う', u'え', u'お'},
			(unichar [ 8]) {u'あ', u'か', u'が', u'さ', u'ざ', u'た', u'だ', u'な'},
			(unichar [ 8]) {u'は', u'ば', u'ぱ', u'ま', u'や', u'ら', u'わ', u'ん'}
		}
	},
	gKeyboardKO =
	{
		3,
		(unsigned int [3]) {10, 10, 8},
		10.0,
		(double [3]) {0.0, 0.0, 1.0},
		(unichar *[3])
		{
			(unichar [10]) {u'ㅂ', u'ㅈ', u'ㄷ', u'ㄱ', u'ㅅ', u'ㅛ', u'ㅕ', u'ㅑ', u'ㅐ', u'ㅔ'},
			(unichar [10]) {u'ㅁ', u'ㄴ', u'ㅇ', u'ㄹ', u'ㅎ', u'ㅗ', u'ㅓ', u'ㅏ', u'ㅣ', KOREAN_ENTER_CHARACTER},
			(unichar [ 8]) {u'ㅋ', u'ㅌ', u'ㅊ', u'ㅍ', u'ㅠ', u'ㅜ', u'ㅡ', KOREAN_SHIFT_CHARACTER}
		}
	},
	gKeyboardNL =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardPT =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardRU =
	{
		3,
		(unsigned int [3]) {11, 11, 9},
		11.0,
		(double [3]) {0.0, 0.0, 1.0},
		(unichar *[3])
		{
			(unichar [11]) {u'й', u'ц', u'у', u'к', u'е', u'н', u'г', u'ш', u'щ', u'з', u'х'},
			(unichar [11]) {u'ф', u'ы', u'в', u'а', u'п', u'р', u'о', u'л', u'д', u'ж', u'э'},
			(unichar [ 9]) {u'я', u'ч', u'с', u'м', u'и', u'т', u'ь', u'б', u'ю'}
		}
	},
	gKeyboardVI =
	{
		3,
		(unsigned int [3]) {10, 9, 7},
		10.0,
		(double [3]) {0.0, 0.5, 1.5},
		(unichar *[3])
		{
			(unichar [10]) {u'q', u'w', u'e', u'r', u't', u'y', u'u', u'i', u'o', u'p'},
			(unichar [ 9]) {u'a', u's', u'd', u'f', u'g', u'h', u'j', u'k', u'l'},
			(unichar [ 7]) {u'z', u'x', u'c', u'v', u'b', u'n', u'm'}
		}
	},
	gKeyboardZH =
	{
		3,
		(unsigned int [3]) {0, 1, 0},
		2.0,
		(double [3]) {0.0, 0.5, 0.0},
		(unichar *[3])
		{
			(unichar [0]) {},
			(unichar [1]) {u' '},	//	placeholder character only
			(unichar [0]) {}
		}
	};


//	Privately-declared properties and methods
@interface TorusGamesKeyboardView()
- (void)touchUpInCrosswordKey:(id)sender;
@end


@implementation TorusGamesKeyboardView
{
	id<TorusGamesKeyboardViewDelegate>	__weak	itsDelegate;
	UIButton									*itsKeys[MAX_ROWS][MAX_COLUMNS];
	bool										itsKoreanKeyboardIsUpshifted;
	UITextField									*itsCharacterReceiver;
	unsigned int								itsNumCharactersReceived;
}


- (id)initWithDelegate:(id<TorusGamesKeyboardViewDelegate>)aDelegate frame:(CGRect)aFrame;
{
	unsigned int	theRow,
					theCol;
	
	self = [super initWithFrame:aFrame];
	if (self != nil)
	{
		itsDelegate = aDelegate;

		for (theRow = 0; theRow < MAX_ROWS; theRow++)
			for (theCol = 0; theCol < MAX_COLUMNS; theCol++)
				itsKeys[theRow][theCol] = nil;

		itsKoreanKeyboardIsUpshifted = false;
		
		//	Create a field to receive character input.
		//
		//		Note #1:  A UITextField can become first responder
		//		even when it's hidden.  Fortunately it doesn't respond
		//		to touches while hidden, so itsCharacterReceiver
		//		becomes first responder only when we explicitly send it
		//		a becomeFirstResponder message.
		//
		//		Note #2:  I tried to implement itsCharacterReceiver
		//		as a subclass of UIResponder that implemented the UIKeyInput protocol
		//		(and also UITextInputTraits, just for good measure),
		//		but contrary to Apple's claim that "When instances of this subclass
		//		are the first responder, the system keyboard is displayed.",
		//		I couldn't get the system keyboard to appear.
		//		Making itsCharacterReceiver a UITextField, and adding it
		//		to the parent view, solved the problem.
		//
		itsCharacterReceiver = [[UITextField alloc] initWithFrame:(CGRect){{0.0, 0.0}, {64.0, 32.0}}];
		[self addSubview:itsCharacterReceiver];
		[itsCharacterReceiver
			addTarget:			self
			action:				@selector(characterReceiverTextDidChange:)
			forControlEvents:	UIControlEventEditingChanged];
		[itsCharacterReceiver setDelegate:self];
		[itsCharacterReceiver setHidden:YES];
		[itsCharacterReceiver setBackgroundColor:[UIColor redColor]];	//	useful while testing
		itsNumCharactersReceived = 0;

		[self setOpaque:NO];
	}
	return self;
}


- (void)refreshWithLanguage:(const Char16 [3])aTwoLetterLanguageCode
{
	unsigned int	theRow,
					theCol,
					theLanguage,
					theTag;
	KeyboardLayout	*theKeyboard;
	UIColor			*theColor,
					*theColorSpecial,	//	for the 5 variable kana keys and the Korean shift and enter keys
					*theColorPlain;		//	for all other keys
	unichar			theCharacter;
	UIButton		*theKey;
	CGFloat			theKeyWidth,
					theKeyHeight;
	CGFloat			theKeyWidth2;
	
	//	Release any pre-existing keys.
	for (theRow = 0; theRow < MAX_ROWS; theRow++)
	{
		for (theCol = 0; theCol < MAX_COLUMNS; theCol++)
		{
			[itsKeys[theRow][theCol] removeFromSuperview];
			itsKeys[theRow][theCol] = nil;
		}
	}
	
	theLanguage = LANGUAGE_CODE(aTwoLetterLanguageCode[0], aTwoLetterLanguageCode[1]);
	
	switch (theLanguage)
	{
		case LANGUAGE_CODE('d','e'): theKeyboard = &gKeyboardDE; break;
		case LANGUAGE_CODE('e','l'): theKeyboard = &gKeyboardEL; break;
		case LANGUAGE_CODE('e','n'): theKeyboard = &gKeyboardEN; break;
		case LANGUAGE_CODE('e','s'): theKeyboard = &gKeyboardES; break;
		case LANGUAGE_CODE('f','i'): theKeyboard = &gKeyboardFI; break;
		case LANGUAGE_CODE('f','r'): theKeyboard = &gKeyboardFR; break;
		case LANGUAGE_CODE('i','t'): theKeyboard = &gKeyboardIT; break;
		case LANGUAGE_CODE('j','a'): theKeyboard = &gKeyboardJA; break;
		case LANGUAGE_CODE('k','o'): theKeyboard = &gKeyboardKO; break;
		case LANGUAGE_CODE('n','l'): theKeyboard = &gKeyboardNL; break;
		case LANGUAGE_CODE('p','t'): theKeyboard = &gKeyboardPT; break;
		case LANGUAGE_CODE('r','u'): theKeyboard = &gKeyboardRU; break;
		case LANGUAGE_CODE('v','i'): theKeyboard = &gKeyboardVI; break;
		case LANGUAGE_CODE('z','s'): theKeyboard = &gKeyboardZH; break;
		case LANGUAGE_CODE('z','t'): theKeyboard = &gKeyboardZH; break;
		
		case LANGUAGE_CODE('-','-'): return;	//	Crossword not active
		
		default: return;	//	should never occur
	}
	
	//	Make sure theKeyboard doesn't overrun the fixed space allowed for itsKeys.
	if (theKeyboard->itsNumRows > MAX_ROWS)
		return;
	for (theRow = 0; theRow < theKeyboard->itsNumRows; theRow++)
		if (theKeyboard->itsRowLengths[theRow] > MAX_COLUMNS)
			return;

	theColorPlain	= [UIColor colorWithRed:0.00 green:0.00 blue:0.50 alpha:1.00];
	theColorSpecial	= [UIColor colorWithRed:1.00 green:0.00 blue:0.00 alpha:1.00];

	theKeyWidth  = [self bounds].size.width  / theKeyboard->itsFullWidth;
	theKeyHeight = [self bounds].size.height / theKeyboard->itsNumRows;
	
	for (theRow = 0; theRow < theKeyboard->itsNumRows; theRow++)
	{
		for (theCol = 0; theCol < theKeyboard->itsRowLengths[theRow]; theCol++)
		{
			//	Technical Note:  Postpone  [self addSubview:theKey]
			//	until after all theKey's properties have been set,
			//	to avoid needless calls to -layoutSubviews.
			
			theKey = [UIButton buttonWithType:UIButtonTypeRoundedRect];
			itsKeys[theRow][theCol] = theKey;

			theCharacter = theKeyboard->itsRows[theRow][theCol];

			//	Even though the Torus Games' Crossword code
			//	works with lowercase characters internally,
			//	let's present uppercase letters to the user
			//	(in languages which make a lowercase/uppercase distinction).
			theCharacter = ToUpperCase(theCharacter);

			//	For the Japanese and Korean text entry systems,
			//	store some useful information in theKey's tag.
			switch (theLanguage)
			{
				case LANGUAGE_CODE('j','a'):

					if (theRow == 0)	//	variable kana
					{
						//	Record the column index, so we can 
						//	recognize theKey in touchUpInCrosswordKey.
						theTag = theCol;
					}
					else
					{
						//	Copy a selector kana into the tag, for convenience
						//	in touchUpInCrosswordKey.  This usage won't conflict
						//	with the column indices written immediately above,
						//	because the column indices run only from 0 to 4.
						//	The only non-selector kana in the list is ん,
						//	which is treated as a normal letter.
						if (theCharacter != u'ん')
							theTag = theCharacter;
						else
							theTag = GENERIC_KEY;
					}

					break;

				case LANGUAGE_CODE('k','o'):

					if (theCharacter == KOREAN_SHIFT_CHARACTER)
						theTag = KOREAN_SHIFT_KEY;
					else
					if (theCharacter == KOREAN_ENTER_CHARACTER)
						theTag = KOREAN_ENTER_KEY;
					else
						theTag = JAMO_KEY;

					break;
				
				case LANGUAGE_CODE('z', 's'):
				case LANGUAGE_CODE('z', 't'):
					theTag = HANZI_REQUEST_KEY;
					break;

				default:
					//	We won't need the tag, so set it to something harmless.
					theTag = GENERIC_KEY;
					break;
			}
			[theKey setTag:theTag];

			//	Let theKey show theCharacter.
			//	(We'll overwrite this for Chinese -- see below.)
			//
			[theKey setTitle:	[NSString stringWithCharacters:&theCharacter length:1]
					forState:	UIControlStateNormal];

			switch (theLanguage)
			{
//				case LANGUAGE_CODE('j','a'):
//					theColor = (theRow == 0 ? theColorSpecial : theColorPlain);
//					break;

				case LANGUAGE_CODE('k','o'):
					theColor = (theCharacter == KOREAN_SHIFT_CHARACTER
							 || theCharacter == KOREAN_ENTER_CHARACTER 
						? theColorSpecial : theColorPlain);
					break;

				default:
					theColor = theColorPlain;
					break;
			}
			[theKey setTitleColor:theColor forState:UIControlStateNormal];

			if (theLanguage == LANGUAGE_CODE('j','a') && theRow == 0)
				theKeyWidth2 = (9.0/5.0) * theKeyWidth;	//	5 wide keys fill the space of 9 normal ones.
			else
				theKeyWidth2 = theKeyWidth;

			[theKey setFrame:(CGRect)
				{
					{
						theKeyWidth*theKeyboard->itsRowStarts[theRow] + theKeyWidth2*theCol + KEYINSET_H,
						theKeyHeight*theRow + KEYINSET_V
					},
					{
						theKeyWidth2 - 2*KEYINSET_H,
						theKeyHeight - 2*KEYINSET_V
					}
				}];

			//	Note:  UIFont caches fonts, so once it's created a font,
			//	we'll keep getting pointers to the same object.
			//	There's no need for us to cache a pointer to it ourselves.

			[[theKey titleLabel] setFont:[UIFont systemFontOfSize:22.0]];

			//	Use a slightly larger font for the variable kana keys.
			if (theLanguage == LANGUAGE_CODE('j','a') && theRow == 0)
				[[theKey titleLabel] setFont:[UIFont boldSystemFontOfSize:24.0]];

			//	Some fonts draw the up-arrow ⇧ thick and some draw it thin.
			//	For the Korean keyboard's shift key, choose a font that
			//	draws it thick.  Hmmm...  AppleGothic should draw a thick
			//	up-arrow, and iOS reports it as present, yet this code
			//	makes no difference in practice.  
			//	Aha!  HiraKakuProN-W6 draws a nice thick ⬆,
			//	even though it draws a thin ⇧.
			if (theCharacter == KOREAN_SHIFT_CHARACTER)
				[[theKey titleLabel] setFont:[UIFont
					fontWithName:	@"HiraKakuProN-W6"
					size:			[[[theKey titleLabel] font] pointSize]]];
						
			[[theKey titleLabel] setTextAlignment:NSTextAlignmentCenter];

			//	For Chinese, the single key calls up the iOS keyboard.
			if (theLanguage == LANGUAGE_CODE('z', 's'))
			{
				//	Let theKey invite the user to enter hanzi (literally "fill in hanzi").
				[theKey setTitle:@"填汉字" forState:UIControlStateNormal];

				//	Set the font to "STHeitiSC-Light" or ""STHeitiSC-Medium".
				[[theKey titleLabel] setFont:[UIFont fontWithName:@"STHeitiSC-Light" size:20.0]];
			}
			if (theLanguage == LANGUAGE_CODE('z', 't'))
			{
				//	Let theKey invite the user to enter hanzi (literally "fill in hanzi").
				[theKey setTitle:@"填漢字" forState:UIControlStateNormal];

				//	Set the font to "STHeitiSC-Light" or ""STHeitiSC-Medium".
				[[theKey titleLabel] setFont:[UIFont fontWithName:@"STHeitiTC-Light" size:20.0]];
			}

//	For use while experimenting with font sizes
//if (theCharacter == 'g' || theCharacter == 'G')
//if (theRow == 0)
//{
// [[theKey titleLabel] setBackgroundColor:[UIColor cyanColor]];
// [theKey setBackgroundColor:[UIColor yellowColor]];
//}

			[theKey
				addTarget:			self
				action:				@selector(touchUpInCrosswordKey:)
				forControlEvents:	UIControlEventTouchUpInside];

			//	[self addSubview:theKey] should come after
			//	[[theKey titleLabel] setTextAlignment:UITextAlignmentCenter],
			//	lest each call to the latter trigger -layoutSubviews for the whole keyboard.
			[self addSubview:theKey];
		}
	}
}


- (void)touchUpInCrosswordKey:(id)sender
{
	UIButton		*theButton;
	unsigned int	theButtonTag;
	NSString		*theButtonTitle;
	unichar			theCharacter,
					*theVariants;
	unsigned int	theCol;
	
	theButton		= (UIButton *) sender;
	theButtonTag	= (unsigned int) [theButton tag];
	theButtonTitle	= [theButton currentTitle];
	if (theButtonTitle != nil && [theButtonTitle length] > 0)
		theCharacter = [theButtonTitle characterAtIndex:0];
	else
		theCharacter = 0;	//	should never occur

	if (theButtonTag == HANZI_REQUEST_KEY)
	{
		//	The user has requested the iOS keyboard to enter hanzi.
		[itsCharacterReceiver becomeFirstResponder];

		//	Don't send the key's own label to the puzzle.
		theCharacter = 0;
	}
	else
	if (theButtonTag < 5)
	{
		//	The user has pressed one of the five variable kana keys in a Japanese puzzle.

		//	Ignore non-existent or obsolete kana (yi, ye, wi, wu, we),
		//	which get displayed to the user as full-width spaces,
		//	but process all other variable kana normally.
		if (theCharacter == u'　')
			theCharacter = 0;
	}
	else
	if (theButtonTag == JAMO_KEY
	 || theButtonTag == KOREAN_SHIFT_KEY
	 || theButtonTag == KOREAN_ENTER_KEY)
	{
		//	Only the shift key will up-shift the letters.
		if ( ! itsKoreanKeyboardIsUpshifted		//	If the down-shifted key are currently active
		 && theButtonTag == KOREAN_SHIFT_KEY)	//	and the user pressed the shift key, then
		{
			//	Insert up-shifted values into the top row.
			[itsKeys[0][0] setTitle:LocalizationNotNeeded(@"ㅃ") forState:UIControlStateNormal];
			[itsKeys[0][1] setTitle:LocalizationNotNeeded(@"ㅉ") forState:UIControlStateNormal];
			[itsKeys[0][2] setTitle:LocalizationNotNeeded(@"ㄸ") forState:UIControlStateNormal];
			[itsKeys[0][3] setTitle:LocalizationNotNeeded(@"ㄲ") forState:UIControlStateNormal];
			[itsKeys[0][4] setTitle:LocalizationNotNeeded(@"ㅆ") forState:UIControlStateNormal];
			[itsKeys[0][8] setTitle:LocalizationNotNeeded(@"ㅒ") forState:UIControlStateNormal];
			[itsKeys[0][9] setTitle:LocalizationNotNeeded(@"ㅖ") forState:UIControlStateNormal];
			
			itsKoreanKeyboardIsUpshifted = true;
		}
		else	//	Any key (Jamo or shift key) may down-shift the letters.
		{
			if (itsKoreanKeyboardIsUpshifted)
			{
				//	Insert down-shifted values into the top row.
				[itsKeys[0][0] setTitle:LocalizationNotNeeded(@"ㅂ") forState:UIControlStateNormal];
				[itsKeys[0][1] setTitle:LocalizationNotNeeded(@"ㅈ") forState:UIControlStateNormal];
				[itsKeys[0][2] setTitle:LocalizationNotNeeded(@"ㄷ") forState:UIControlStateNormal];
				[itsKeys[0][3] setTitle:LocalizationNotNeeded(@"ㄱ") forState:UIControlStateNormal];
				[itsKeys[0][4] setTitle:LocalizationNotNeeded(@"ㅅ") forState:UIControlStateNormal];
				[itsKeys[0][8] setTitle:LocalizationNotNeeded(@"ㅐ") forState:UIControlStateNormal];
				[itsKeys[0][9] setTitle:LocalizationNotNeeded(@"ㅔ") forState:UIControlStateNormal];

				itsKoreanKeyboardIsUpshifted = false;
			}
		}
		
		//	Don't send the shift key itself to the puzzle.
		if (theButtonTag == KOREAN_SHIFT_KEY)
			theCharacter = 0;
		
		//	Let the enter key send a newline.
		if (theButtonTag == KOREAN_ENTER_KEY)
			theCharacter = u'\n';
	}
	else
	if (theButtonTag < GENERIC_KEY)
	{
		//	The user has pressed one of the 15 selector keys in a Japanese puzzle.
		//	Set the five variable kana accordingly.
		switch (theButtonTag)
		{
			case u'あ': theVariants = (unichar [5]) {u'あ', u'い', u'う', u'え', u'お'}; break;
			case u'か': theVariants = (unichar [5]) {u'か', u'き', u'く', u'け', u'こ'}; break;
			case u'が': theVariants = (unichar [5]) {u'が', u'ぎ', u'ぐ', u'げ', u'ご'}; break;
			case u'さ': theVariants = (unichar [5]) {u'さ', u'し', u'す', u'せ', u'そ'}; break;
			case u'ざ': theVariants = (unichar [5]) {u'ざ', u'じ', u'ず', u'ぜ', u'ぞ'}; break;
			case u'た': theVariants = (unichar [5]) {u'た', u'ち', u'つ', u'て', u'と'}; break;
			case u'だ': theVariants = (unichar [5]) {u'だ', u'ぢ', u'づ', u'で', u'ど'}; break;
			case u'な': theVariants = (unichar [5]) {u'な', u'に', u'ぬ', u'ね', u'の'}; break;
			case u'は': theVariants = (unichar [5]) {u'は', u'ひ', u'ふ', u'へ', u'ほ'}; break;
			case u'ば': theVariants = (unichar [5]) {u'ば', u'び', u'ぶ', u'べ', u'ぼ'}; break;
			case u'ぱ': theVariants = (unichar [5]) {u'ぱ', u'ぴ', u'ぷ', u'ぺ', u'ぽ'}; break;
			case u'ま': theVariants = (unichar [5]) {u'ま', u'み', u'む', u'め', u'も'}; break;
			case u'や': theVariants = (unichar [5]) {u'や', u'　', u'ゆ', u'　', u'よ'}; break;
			case u'ら': theVariants = (unichar [5]) {u'ら', u'り', u'る', u'れ', u'ろ'}; break;
			case u'わ': theVariants = (unichar [5]) {u'わ', u'　', u'　', u'　', u'を'}; break;
			default:   theVariants = (unichar [5]) {u'？', u'？', u'？', u'？', u'？'}; break;
		}
		for (theCol = 0; theCol < 5; theCol++)
		{
			[itsKeys[0][theCol] setTitle:
				[NSString stringWithCharacters:&theVariants[theCol] length:1]
				forState:UIControlStateNormal];
		}
		
		//	Don't send the selector itself to the puzzle.
		theCharacter = 0;
	}
	else
	{
		//	The user has pressed a key in a non-Japanese puzzle.
		//	Process theCharacter normally.
	}

	if (theCharacter != 0)
		[itsDelegate userTappedKey:theCharacter];
}


//	UITextField notification

- (void)characterReceiverTextDidChange:(id)sender
{
	NSString		*theText;
	UITextRange		*theMarkedTextRange;
	NSInteger		theMarkedTextStart,
					theMarkedTextEnd;
	NSUInteger		theCompletedTextLength;
	
	if (sender != itsCharacterReceiver)	//	should never occur
		return;

	//	itsCharacterReceiver's text will typically consist of a string
	//	of completed hanzi, followed by some provisional "composing text".
	//
	//	Example #1:  In the Pinyin IME, writing "中文" might give the sequence
	//
	//			[z]
	//			[zh]
	//			[zho]
	//			中
	//			中[w]
	//			中[we]
	//			中[wen]
	//			中文
	//
	//	where the characters in […] are marked as provisionally inserted text.
	//
	//	Example #2:  In the Handwriting IME, writing "三十" might give
	//
	//			[一]
	//			[二]
	//			[三]
	//			三
	//			三[一]
	//			三[十]
	//			三十
	//
	//	At each step we'd like to read out any newly completed hanzi,
	//	while ignoring the temporary composing characters.

	theText = [itsCharacterReceiver text];
//NSLog(@"modified text: %@", theText);

	theMarkedTextRange = [itsCharacterReceiver markedTextRange];
	if (theMarkedTextRange != nil)
	{
		theMarkedTextStart	= [itsCharacterReceiver
								offsetFromPosition:	[itsCharacterReceiver beginningOfDocument]
								toPosition:			[theMarkedTextRange start]];
		theMarkedTextEnd	= [itsCharacterReceiver
								offsetFromPosition:	[itsCharacterReceiver beginningOfDocument]
								toPosition:			[theMarkedTextRange end]];

		theCompletedTextLength = theMarkedTextStart;
		
		(void)theMarkedTextEnd;	//	Let the static analyzer know that theMarkedTextEnd isn't used.
		
		//	If we wanted to somehow display the provisional text
		//	(the "marked text") we could, but we don't.
		//	The platform-independent code maintains itsPendingCharacter
		//	for situations in which the platform-independent code itself
		//	must compose the characters, for example
		//
		//			':' + 'u' = 'ü'
		//	and
		//			's' + 'u' = 'す'
		//
		//	In the present situation, the IME will ultimately provide
		//	the completed character on its own, so we can safely ignore
		//	the provisional text.
	}
	else	//	There is no marked text.
	{
		theCompletedTextLength = [theText length];
	}
	
	//	If the user deletes previously completed characters,
	//	send the platform-independent code a backspace for each one.
	//	(It's hard to imagine more than one character could
	//	be deleted at a time, but this code doesn't assume that.)
	while (itsNumCharactersReceived > theCompletedTextLength)
	{
		[itsDelegate userTappedKey:'\b'];

		itsNumCharactersReceived--;
	}

	//	Has one or more newly completed character arrived?
	while (itsNumCharactersReceived < theCompletedTextLength)
	{
		[itsDelegate userTappedKey:[theText characterAtIndex:itsNumCharactersReceived]];
		
		itsNumCharactersReceived++;
	}
}

//	UITextFieldDelegate

- (BOOL)textField:(UITextField *)textField shouldChangeCharactersInRange:(NSRange)range replacementString:(NSString *)string
{
	return YES;
}

- (void)textFieldDidEndEditing:(UITextField *)textField
{
	//	Now's a good moment to clear any accumulated text,
	//	to keep it from building up indefinitely.
	[itsCharacterReceiver setText:@""];
	itsNumCharactersReceived = 0;
}

- (BOOL)textFieldShouldReturn:(UITextField *)textField
{
	[itsCharacterReceiver resignFirstResponder];
	return NO;
}


@end
